#!/usr/bin/env python

#createList.py --yT 46 --zT 150 --fidList overflow.txt --overflowLog overflow_log_dru1.txt 

import re, os, sys
from optparse import OptionParser

usage   = 'usage: %prog [options]'
descr   = 'create a sampling list that excludes corrupted FIDs'
vinfo   = '%prog version 1.0'

parser = OptionParser(usage = usage, description=descr, version=vinfo)

parser.add_option('-y', '--yT',
                   action  = 'store',
                   dest    = 'yT',
                   type    = 'int',
                   default = 1,
                   help    = 'No. of y-increments, i.e. -yT in fid.com')
parser.add_option('-z', '--zT',
                   action  = 'store',
                   dest    = 'zT',
                   type    = 'int',
                   default = 1,
                   help    = 'No. of z-increments, i.e. -zT in fid.com')
parser.add_option('-a', '--aT',
                   action  = 'store',
                   dest    = 'aT',
                   type    = 'int',
                   default = 1,
                   help    = 'No. of a-increments, i.e. -aT in fid.com')
parser.add_option('-f', '--fidList',
                   action  = 'store',
                   dest    = 'uFidList',
                   default = 'None',
                   help    = 'User list file for the corrupted FIDs')
parser.add_option('-o', '--overflowLog',
                   action  = 'store',
                   dest    = 'tsOFLog',
                   default = 'overflow_log_dru1.txt',
                   help    = 'Topspin log file for the FIDs with DRU overflows')

(opts, args) = parser.parse_args()
yT = opts.yT
zT = opts.zT
aT = opts.aT

uFidList = opts.uFidList
tsOFLog  = opts.tsOFLog

yN = zN = aN = 1
if aT > 1:
   aN = aT*2
if zT > 1:
   zN = zT*2
if yT > 1:
   yN = yT*2

if aT > 1:
   nusDimCount = 3
elif zT > 1:
   nusDimCount = 2
elif yT > 1:
   nusDimCount = 1
else:
   sys.stderr.write("Incorrect input matrix size. Invalid number of NUS dimension!\n\n")
   exit()

totNumFIDs = yN*zN*aN
fidNumList = []

if uFidList != 'None':
   if os.path.isfile(uFidList):
      fileUnit  = open(uFidList, 'r')
      linesRead = fileUnit.readlines()
      fileUnit.close()

      #put the bad FIDs into a list
      for line in linesRead:
         if re.search('-', line):
            overflowList = line.split('-')
            startFID  = int(overflowList[0])
            endFID    = int(overflowList[1].strip()) + 1

            if startFID > totNumFIDs or endFID > totNumFIDs+1:
               sys.stderr.write('Invalid FID number found!')
               exit()

            fidNumList.extend(range(startFID, endFID))
         else:
            if int(line.strip()) > totNumFIDs:
               sys.stderr.write('Invalid FID number found!')
               exit()

            fidNumList.append(int(line.strip()))
   else:
      sys.stderr.write('User list file %s for corrupted FIDs not found.\n\n' % (uFidList))

#read the bad FID number if Topspin overflow log file overflow_log_dru1.txt exists
if os.path.isfile(tsOFLog):
   fileUnit  = open(tsOFLog)
   linesRead = fileUnit.readlines()
   fileUnit.close()

   #add the FIDs with DRU overflow into the list
   for line in linesRead:
      if line[0] == "#":
         continue

      tmp = line.split()
      if len(tmp) != 3:
         continue

      if int(tmp[1]) > totNumFIDs:
         sys.stderr.write('Invalid FID number found!')
         exit()

      if int(tmp[1]) not in fidNumList:
         fidNumList.append(int(tmp[1]))

if not fidNumList:
   sys.stderr.write('There are no FIDs to be excluded!\n\n')
   sys.stderr.write('No NUS sampling list is generated!\n\n')
   exit()

#convert each FID No. into a tuple of 3D zero-based increment numbers
#the tuple is then added to an exclusion list
xList = []
for fid in fidNumList:
   ia = (fid-1)/(yN*zN*2)
   im = (fid-1)%(yN*zN*2)
   im = im%(yN*zN)
   iz = im/(yN*2)
   im = im%(yN*2)
   iy = im%yN/2

   if (iy, iz, ia) not in xList:
      xList.append((iy, iz, ia))

#output the sampling points not included in the exclusion list
for ia in range(aT):
   for iz in range(zT):
      for iy in range(yT):
         if (iy, iz, ia) not in xList:
            if nusDimCount == 3:
               print("%3d %3d %3d" % (iy, iz, ia))
            elif nusDimCount == 2:
               print("%3d %3d" % (iy, iz))
            elif nusDimCount == 1:
               print("%3d" % (iy))


